<?php
/**
 * Functions
 *
 * @package     AutomatorWP\QR_Code\Functions
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

use BaconQrCode\Writer;
use BaconQrCode\Renderer\Color\Rgb;
use BaconQrCode\Renderer\Eye\SquareEye;
use BaconQrCode\Renderer\Image\ImagickImageBackEnd;
use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use BaconQrCode\Renderer\Image\EpsImageBackEnd;
use BaconQrCode\Renderer\ImageRenderer;
use BaconQrCode\Renderer\Module\SquareModule;
use BaconQrCode\Renderer\RendererStyle\EyeFill;
use BaconQrCode\Renderer\RendererStyle\Fill;
use BaconQrCode\Renderer\RendererStyle\Gradient;
use BaconQrCode\Renderer\RendererStyle\GradientType;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;
use BaconQrCode\Renderer\GDLibRenderer;
use BaconQrCode\Common\ErrorCorrectionLevel;
use BaconQrCode\Encoder\Encoder;

//require_once AUTOMATORWP_QR_CODE_DIR . 'vendor/autoload.php';

/**
 * Get writer for gradient color.
 *
 * @param string $format    QR code format.
 *
 * @return WriterInterface 
 */
function automatorwp_qr_code_get_gradient_writer( $format ) {

    switch ( $format ) {
		case 'svg':
			return new SvgImageBackEnd();
		case 'eps':
			return new EpsImageBackend();
		case 'png':
		default:
			return new ImagickImageBackEnd();
	}
}

/**
 * Convert color to RGB.
 *
 * @param string $color    HEX Color.
 *
 * @return array
 */
function automatorwp_qr_code_hex_to_rgb( $color ) {

	if ( $color[0] == '#' ) {
		$color = substr( $color, 1 );
	}

	if ( strlen( $color ) == 6 ) {
		list( $r, $g, $b ) = array( $color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5] );
	} elseif ( strlen( $color ) == 3 ) {
		list( $r, $g, $b ) = array( $color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2] );
	} else {
			return false;
	}

	$r = hexdec( $r );
	$g = hexdec( $g );
	$b = hexdec( $b );

	return array( 'red' => $r, 'green' => $g, 'blue' => $b );

}

/**
 * Get correction level.
 *
 * @param string $level    Correction error level.
 *
 * @return object
 */
function automatorwp_qr_code_get_correction_level( $level ) {

	switch ( $level ) {
		case 'low':
			$level = ErrorCorrectionLevel::L();
			break;
		case 'medium':
			$level = ErrorCorrectionLevel::M();
			break;
		case 'quartile':
			$level = ErrorCorrectionLevel::Q();
			break;
		case 'high':
			$level = ErrorCorrectionLevel::H();
			break;
		default:
			$level = ErrorCorrectionLevel::L();
	}

	return $level;

}

/**
 * Create QR code with single/gradient color.
 *
 * @since 1.0.0
 * 
 * @param string $qr_content    			Content(string, URL).
 * @param int	 $qr_size    				Size.
 * @param string $qr_foreground_color_1    	Foreground color 1.
 * @param string $qr_foreground_color_2    	Foreground color 2.
 * @param string $qr_foreground_eye_color_1    	Foreground eye color 1.
 * @param string $qr_foreground_eye_color_2    	Foreground eye color 2.
 * @param string $qr_background_color    	Background color.
 * @param string $qr_error_level    		Correction error level.
 * @param string $qr_format    				PNG, SVG, EPS.
 * @param string $qr_code_path    			QR code path.
 * @param string $qr_set_eye    			Eye color on/off.
 *
 */
function automatorwp_qr_code_generate_gradient_color( $qr_content, $qr_size, $qr_foreground_color_1, $qr_foreground_color_2, $qr_foreground_eye_color_1, $qr_foreground_eye_color_2, $qr_background_color, $qr_error_level, $qr_format, $qr_code_path, $qr_set_eye = false ) {
	
	$eye = SquareEye::instance();
    $squareModule = SquareModule::instance();

    $eyeFill = new EyeFill( new Rgb( (int) $qr_foreground_eye_color_1['red'], (int) $qr_foreground_eye_color_1['green'], (int) $qr_foreground_eye_color_1['blue'] ) , new Rgb( (int) $qr_foreground_eye_color_2['red'], 
										(int) $qr_foreground_eye_color_2['green'], (int) $qr_foreground_eye_color_2['blue'] ) );
    $gradient = new Gradient(new Rgb( (int) $qr_foreground_color_1['red'], (int) $qr_foreground_color_1['green'], (int) $qr_foreground_color_1['blue'] ) , new Rgb( (int) $qr_foreground_color_2['red'], 
										(int) $qr_foreground_color_2['green'], (int) $qr_foreground_color_2['blue'] ), GradientType::HORIZONTAL());

	if ( (bool) $qr_set_eye ) {	
		$qr_gradient = Fill::withForegroundGradient(new Rgb( (int) $qr_background_color['red'], (int) $qr_background_color['green'], (int) $qr_background_color['blue'] ), $gradient, $eyeFill, $eyeFill, $eyeFill);		
	} else {
		$qr_gradient = Fill::uniformGradient(new Rgb( (int) $qr_background_color['red'], (int) $qr_background_color['green'], (int) $qr_background_color['blue'] ), $gradient);		
	}

    $renderer = new ImageRenderer(
        new RendererStyle(
            $qr_size,
            2,
            $squareModule,
            $eye,
			$qr_gradient,
        ),
        automatorwp_qr_code_get_gradient_writer( $qr_format )
    );
	
    $writer = new Writer( $renderer );
	  
	$writer->writeFile( $qr_content, $qr_code_path, Encoder::DEFAULT_BYTE_MODE_ECODING, $qr_error_level );
	
}

/**
 * Create standard QR code.
 *
 * @since 1.0.0
 * 
 * @param string $qr_content    			Content(string, URL).
 * @param int	 $qr_size    				Size.
 * @param string $qr_code_path    			QR code path.
 *
 */
function automatorwp_qr_code_generate_standard_qr( $qr_content, $qr_size, $qr_code_path ) {

	$renderer = new GDLibRenderer( $qr_size );
	$writer = new Writer( $renderer );
	   
	$writer->writeFile( $qr_content, $qr_code_path );

}